
using Godot;


namespace Rokojori
{  
  [Tool]
  [GlobalClass, Icon("res://addons/rokojori_action_library/Icons/WindManager.svg") ]
  public partial class WindManager:Node
  {   
    [Export]
    public WindManagerData data;

    [Export]
    public Action onWindChange;

     [Export]
    public Action onWindDirectionChange;
    
    [Export]
    public Action onWindSpeedChange;

    float _currentKMH = 0;
    Vector2 _currentDirection = Vector2.Zero;

    public override void _Process( double delta )
    {
      var rm = Unique<RenderingManager>.Get();

      if ( data == null || rm == null )
      {
        return;
      }

      UpdatePosition( rm, (float) delta, data.globalWindPositionClosePropertyName, data.closeMultiplier );
      UpdatePosition( rm, (float) delta, data.globalWindPositionFarPropertyName, data.farMultiplier );
      UpdateDirection( rm, (float) delta );
      UpdateSpeed( rm, (float) delta );

      var changed = false;

      if ( _currentKMH != data.windSpeed.GetKMH() )
      {
        changed = true;
        onWindSpeedChange?.Trigger();

        _currentKMH = data.windSpeed.GetKMH();
      }

      if ( _currentDirection != data.windDirection )
      {
        changed = true;
        onWindDirectionChange?.Trigger();

        _currentDirection = data.windDirection;
      }

      if ( changed )
      {
        onWindChange?.Trigger();
      }
    }


    void UpdatePosition( RenderingManager rm, float delta, ShaderPropertyName positionName, float multiplier )
    {
      var windPositionProperty = rm.data.GetGlobalPropertyByName( positionName );

      if ( windPositionProperty == null )
      {
        return;
      }

      var positionProperty = (Vector2Property) windPositionProperty;
      var position = positionProperty.value;
      position += multiplier * -data.windSpeed.GetNormalizedWindSpeedOffset( delta, data.minimumScrollSpeedKMH, data.maximumScrollSpeedKMH )  * data.windDirection.Normalized();
      positionProperty.value = position;
    }

    void UpdateDirection( RenderingManager rm, float delta )
    {
      var windDirectionProperty = rm.data.GetGlobalPropertyByName( data.globalWindDirectionPropertyName );

      if ( windDirectionProperty == null )
      {
        return;
      }

      var directionProperty = (Vector2Property) windDirectionProperty;
      directionProperty.value = data.windDirection.Normalized();
    }

     void UpdateSpeed( RenderingManager rm, float delta )
    {
      var windSpeedProperty = rm.data.GetGlobalPropertyByName( data.globalWindSpeeedPropertyName );

      if ( windSpeedProperty == null )
      {
        return;
      }

      var speedProperty = (FloatProperty) windSpeedProperty;
      speedProperty.value = data.windSpeed.GetKMH();
    }
  }
}