
using Godot;
using System.Threading.Tasks;


namespace Rokojori
{  
  [Tool]
  [GlobalClass]
  public partial class MainModule: PresetModule
  { 
    [ExportGroup("Basic")]
    [Export]
    public bool localeManager = true;

    [Export]
    public bool sensorManager = true;

    [Export]
    public bool timeLineManager = true;

    [Export]
    public bool audioManager = true;

    [Export]
    public bool renderingManager = true;


    [ExportGroup("Camera")]
    [Export]  
    public bool cameraManager = true;

    [Export]
    public bool mainCamera = true;

    [Export]
    public bool debugEditorCamera = true;

    [Export]
    public bool worldEnvironment = true;


    
    public async Task ProcessPass( PresetContext context )
    {
      if ( PresetPass.Process != context.pass )
      {
        return;
      }

      CreateBasic( context );

      CreateCamera( context );

      return;
    }

    void CreateBasic( PresetContext context )
    {
      var root = context.root;

      if ( localeManager )
      {
        root.CreateChild<LocaleManager>( "Locale Manager" );
      }

      if ( sensorManager )
      {
        root.CreateChild<SensorManagerSetup>( "Sensor Manager Setup" );
      }

      if ( timeLineManager )
      {
        var tm = root.CreateChild<TimeLineManager>( "Timeline Manager" );

        var timelinePath = "res://addons/rokojori_action_library/Runtime/Time/TimeLines/";
        var gameTime = ResourceLoader.Load<TimeLine>( timelinePath + "GameTime.tres" );
        var realTime = ResourceLoader.Load<TimeLine>( timelinePath + "RealTime.tres" );

        tm.gametimeTimeline = gameTime;
        tm.realtimeTimeline = realTime;

        tm.timeLines = [ gameTime, realTime ];

      }

      if ( audioManager )
      {
        root.CreateChild<AudioManager>( "Audio Manager" );
      }

      if ( renderingManager )
      {
        root.CreateChild<RenderingManager>( "Rendering Manager" );
      }
    }

    void CreateCamera( PresetContext context )
    {
      var hasCameraSetup = cameraManager || worldEnvironment || mainCamera || debugEditorCamera;

      if ( ! hasCameraSetup )
      {
        return;
      }

      var root = context.root;

      var cameraSetup = root.CreateChild<CameraSetupIcon>( "Camera Setup" );


      CameraManager camManager = null;

      if ( cameraManager )
      {
        camManager = cameraSetup.CreateChild<CameraManager>( "Camera Manager" );
      }

      if ( worldEnvironment )
      {
        var env = cameraSetup.CreateChild<WorldEnvironment>();

        env.Environment = new Environment();
        env.Environment.BackgroundMode = Environment.BGMode.Sky;
        env.Environment.Sky = new Sky();
        env.Environment.Sky.SkyMaterial = new ProceduralSkyMaterial();

        if ( camManager != null )
        {
          camManager.worldEnvironment = env;
        }

        var camAtts = new CameraAttributesPractical();
        env.CameraAttributes = camAtts;
        camAtts.DofBlurNearEnabled = true;
        camAtts.DofBlurNearDistance = 0.2f;
        camAtts.DofBlurNearTransition = 0.1f;
        camAtts.DofBlurAmount = 0.25f;
        
      }

      if ( mainCamera )
      {
        var mainCam = cameraSetup.CreateChild<Camera3D>( "Main Camera" );

        if ( camManager != null )
        {
          camManager.camera = mainCam;
        }
      }

      if ( debugEditorCamera )
      {
        var editorCamera = cameraSetup.CreateChild<EditorCamera>( "Debug Editor Camera" );

        if ( camManager != null )
        {
          camManager.debugCamera = editorCamera;
          var toggleSensor = new KeySensor();
          toggleSensor.key = Key.F5;
          camManager.debugCameraToggle = toggleSensor;        
        }
      }

      


    }
    

  }
}